<?php

namespace App\Http\Controllers\Admin;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Deal;
use App\Models\DealDocuments;
use Illuminate\Support\Facades\Validator;
use App\Models\Category; // Import the Category model at the top of your controller
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class DealsController extends Controller {

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct() {
        $this->middleware('auth');
        $this->middleware('role:admin'); // Ensure only admin users can access this controller
    }

    /**
     * Show a list of deals.
     *
     * @return \Illuminate\View\View
     */
    public function index() {
        // $deals = Deal::all();
        $deals = Deal::withCount(['dealInvestments as total_investors' => function ($query) {
                        $query->select(DB::raw('count(*)'))->groupBy('deal_id');
                    }])->paginate(10); // 10 records per page, adjust as needed
        return view('admin.deals.list', compact('deals'));
    }

    /**
     * Show the form for creating a new deal.
     *
     * @return \Illuminate\View\View
     */
    public function create() {
        $categories = Category::all(); // Retrieve all categories from the database

        return view('admin.deals.add', compact('categories'));
    }

    /**
     * Store a newly created deal in the database.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request) {
        $validator = Validator::make($request->all(), [
                    'deal_name' => 'required|string|max:255',
                    'total_investment' => 'required|numeric',
                    'category_id' => 'required|exists:categories,id',
                    'start_date' => 'required|date',
                    'end_date' => 'required|date|after_or_equal:start_date'
        ]);

        if ($validator->fails()) {
            return redirect()->route('admin.deals.create')
                            ->withErrors($validator)
                            ->withInput();
        }

        // Attempt to create a new deal
        $newdeal = Deal::create($request->all());
        //echo $newdeal;exit;
        if ($newdeal) {
            if ($request->hasFile('document_files')) {
                $uploadedFiles = $request->file('document_files');
                $documentNames = $request->input('document_name');

                foreach ($uploadedFiles as $key => $file) {
                    // Generate a unique filename
                    $filename = time() . '_' . uniqid('document_') . '.' . $file->getClientOriginalExtension();

                    // Define the destination directory
                    $destinationPath = public_path('uploads/deals/documents');

                    // Use the move method to save the file to the destination directory
                    $file->move($destinationPath, $filename);

                    // Create a new DealDocument record for each file with its respective document name
                    DealDocuments::create([
                        'deal_id' => $newdeal->id,
                        'document_name' => $documentNames[$key],
                        'document_file' => $filename,
                        'status' => 1,
                    ]);
                }
                
                
            }
            return redirect()->route('admin.deals.list')->with('success', 'Deal created successfully');
        } else {
            return redirect()->route('admin.deals.create')->with('error', 'Failed to create the deal.');
        }
    }

    /**
     * Show the form for editing the specified deal.
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function edit($id) {
        $deal = Deal::findOrFail($id);
        $categories = Category::all();
        return view('admin.deals.edit', compact('deal', 'categories'));
    }

    /**
     * Update the specified deal in the database.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $id) {
        $validator = Validator::make($request->all(), [
                    'deal_name' => 'required|string|max:255',
                    'total_investment' => 'required|numeric',
                    'category_id' => 'required|exists:categories,id',
                    'start_date' => 'required|date',
                    'end_date' => 'required|date|after_or_equal:start_date',
        ]);

        if ($validator->fails()) {
            return redirect()->route('admin.deals.edit', $id)
                            ->withErrors($validator)
                            ->withInput();
        }

        $deal = Deal::findOrFail($id);
        $dealUpdated = $deal->update($request->all());
        
        if ($dealUpdated) {
            if ($request->hasFile('document_files')) {
                $uploadedFiles = $request->file('document_files');
                $documentNames = $request->input('document_name');

                foreach ($uploadedFiles as $key => $file) {
                    // Generate a unique filename
                    $filename = time() . '_' . uniqid('document_') . '.' . $file->getClientOriginalExtension();

                    // Define the destination directory
                    $destinationPath = public_path('uploads/deals/documents');

                    // Use the move method to save the file to the destination directory
                    $file->move($destinationPath, $filename);

                    // Create a new DealDocument record for each file with its respective document name
                    DealDocuments::create([
                        'deal_id' => $id,
                        'document_name' => $documentNames[$key],
                        'document_file' => $filename,
                        'status' => 1,
                    ]);
                }
                
                
            }
            return redirect()->route('admin.deals.list')->with('success', 'Deal created successfully');
        } else {
            return redirect()->route('admin.deals.create')->with('error', 'Failed to create the deal.');
        }

        return redirect()->route('admin.deals.list')->with('success', 'Deal updated successfully');
    }

    /**
     * Remove the specified deal from the database.
     *
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($id) {
        $deal = Deal::findOrFail($id);
        $deal->delete();
        return redirect()->route('admin.deals.list')->with('success', 'Deal deleted successfully');
    }
    
    public function destroyDealDocument($id) {
        $dealDoc = DealDocuments::findOrFail($id);
        $dealDoc->delete();
        return redirect()->back()->with('success', 'Document deleted successfully');
    }
    

    /**
     * Show the form for creating adding deal document.
     *
     * @return \Illuminate\View\View
     */
    public function documents() {
        $deals = Deal::all(); // Retrieve all categories from the database

        return view('admin.deals.document', compact('deals'));
    }

    /**
     * adding deal document in db.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function storDocuments(Request $request) {
        // Validate the form data
        $request->validate([
            'deal_id' => 'required|exists:deals,id', // Ensure the deal_id exists in the deals table
            'document_name.*' => 'required|string|max:255', // Use ".*" to validate each element in the array
            'document_files.*' => 'required|file|mimes:pdf', // Use ".*" to validate each element in the array
        ]);

        // Process and store each uploaded file
        if ($request->hasFile('document_files')) {
            $uploadedFiles = $request->file('document_files');
            $documentNames = $request->input('document_name');

            foreach ($uploadedFiles as $key => $file) {
                // Generate a unique filename
                $filename = time() . '_' . uniqid('document_') . '.' . $file->getClientOriginalExtension();

                // Define the destination directory
                $destinationPath = public_path('uploads/deals/documents');

                // Use the move method to save the file to the destination directory
                $file->move($destinationPath, $filename);

                // Create a new DealDocument record for each file with its respective document name
                DealDocuments::create([
                    'deal_id' => $request->input('deal_id'),
                    'document_name' => $documentNames[$key], // Use the document name from the array
                    'document_file' => $filename,
                    'status' => 1, // You can set the default status here
                ]);
            }
        }

        // Redirect back with a success message
        return redirect()->route('admin.deals.list')->with('success', 'Documents added successfully');
    }

    public function dealDetails($id) {
        $deal = Deal::with('dealDocuments')->findOrFail($id);

        // Get the investments for the deal and eager load the associated users
        $investments = $deal->dealInvestments()->with('user')->get();

        $dealDocuments = $deal->dealDocuments;

        return view('admin.deals.details', compact('deal', 'dealDocuments', 'investments'));
    }

    public function downloadDocument($filename, $filetyp) {
        // Define the directory where the PDFs are stored
        if($filetyp == 'tax'){
            $directory = 'uploads/deals/taxes';
        }else{
            $directory = 'uploads/deals/documents';
        }

        // Ensure that the filename is safe and sanitize it
        $filename = Str::of($filename)->basename();

        // Define the full path to the PDF file
        $filePath = public_path($directory . '/' . $filename);

        // Check if the file exists
        if (file_exists($filePath)) {
            // Determine the MIME type (PDF)
            $mimeType = 'application/pdf';

            // Generate a unique name for the downloaded file
            $downloadFilename = 'document_' . time() . '.pdf';

            // Set headers to force download
            header("Cache-Control: public");
            header("Content-Description: File Transfer");
            header("Content-Disposition: attachment; filename=$downloadFilename");
            header("Content-Type: $mimeType");
            header("Content-Transfer-Encoding: binary");

            // Read the file and output it to the response
            readfile($filePath);
        } else {
            // File not found, you can handle this case as needed
            abort(404, 'File not found');
        }
    }

    public function updateStatus(Request $request) {
        $dealId = $request->input('deal_id'); // Assuming you have a deal_id in your form
        $newStatus = $request->input('new_status');

        try {
            $deal = Deal::findOrFail($dealId);

            // Update the status field in the database
            $deal->status = $newStatus;
            $deal->save();

            return response()->json(['message' => 'Deal status updated successfully'], 200);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Failed to update deal status'], 500);
        }
    }
}
